<?php
/**
 * Endpoint interne sécurisé (HMAC) pour exécuter les tools MCP.
 * URL: index.php?fc=module&module=iassistant&controller=api
 */
class IassistantApiModuleFrontController extends ModuleFrontController
{
    public $ssl = true;

    public function initContent()
    {
        parent::initContent();
        header('Content-Type: application/json');

        try {
            $this->requirePostJson();

            $rawBody = Tools::file_get_contents('php://input');
            $data = json_decode($rawBody, true, 512, JSON_THROW_ON_ERROR);

            $this->verifySignature($rawBody, $data);

            $tool = (string)($data['tool'] ?? '');
            $args = (array)($data['args'] ?? []);

            $this->assertAllowedTool($tool);

            require_once _PS_MODULE_DIR_.'iassistant/src/Service/McpExecutor.php';
            require_once _PS_MODULE_DIR_.'iassistant/src/Mcp/McpTools.php';

            $exec = new \Iassistant\Service\McpExecutor($this->context);
            $result = $exec->execute($tool, $args);

            $this->log('tool_ok', ['tool'=>$tool]);
            die(json_encode(['ok'=>true, 'result'=>$result]));
        } catch (\Throwable $e) {
            $this->log('tool_err', ['msg'=>$e->getMessage()]);
            http_response_code(400);
            die(json_encode(['ok'=>false, 'error'=>$e->getMessage()]));
        }
    }

    private function requirePostJson(): void
    {
        if (strtoupper($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
            throw new \RuntimeException('POST required');
        }
        if (stripos($_SERVER['CONTENT_TYPE'] ?? '', 'application/json') === false) {
            throw new \RuntimeException('Content-Type application/json required');
        }
    }

    // TO REMOVE FOR PROD
    private function allowDebugBypass($data) {
        if (isset($data['debug']) && $data['debug'] === "WhyJ") {
            return true;
        }
        return false;
    }

    private function verifySignature(string $rawBody, array $data): void
    {

        if ($this->allowDebugBypass($data)) {
            return;
        }

        $ts   = (int)($data['ts'] ?? 0);
        $nonce= (string)($data['nonce'] ?? '');
        $sig  = (string)($_SERVER['HTTP_X_LLM_SIGNATURE'] ?? '');

        if (!$ts || !$nonce || !$sig) {
            throw new \RuntimeException('Missing ts/nonce/signature');
        }
        if (abs(time() - $ts) > 300) {
            throw new \RuntimeException('Timestamp drift too large');
        }

        $secret = Configuration::get('IASSISTANT_SHARED_SECRET');
        // On signe "ts.nonce.RAW_BODY" côté API ; on recalcule pareil.
        $base = $ts . '.' . $nonce . '.' . $rawBody;
        $calc = hash_hmac('sha256', $base, $secret);

        if (!hash_equals($calc, $sig)) {
            throw new \RuntimeException('Bad signature');
        }
    }

    private function assertAllowedTool(string $tool): void
    {
        // IMPORTANT : $tool doit déjà être normalisé (strtolower + '-' -> '_')
        $allowed = [
            // ==== IMAGES ====
            'list_product_images',
            'get_product_image_base64',
            'save_generated_image',
            'regenerate_product_image_thumbnails',
            'set_product_cover_image',
            'set_product_cover_from_first',
            'set_product_cover',                   // alias dédié
            'generate_new_image_from_existing',
            'generate_product_image_from_existing',// alias accepté
            'generate_product_image',              // alias accepté
            'generate_new_image_by_query',

            // ==== LANGUES ====
            'list_languages',
            'get_lang_id',

            // ==== PRODUITS & IMPORT ====
            'create_product',
            'add_product_images_base64',

            // ==== RECHERCHE PRODUIT ====
            'find_product',
            'find_product_by_name',

            // ==== ANALYSE & ACTIONS GROUPEES ====
            'find_low_selling_products',
            'apply_product_recommendations',

            // ==== ACTIONS UNITAIRES PRODUIT ====
            'set_product_active',
            'set_product_price',
            'update_product_name',
            'update_product_description',
            'update_product_short_description',
            'add_product_tags',
            'create_specific_price',
            'ensure_variant_images',
            'review_price_positioning',
            'request_new_images',

            // ==== BACK-OFFICE NAVIGATION ====
            'resolve_backoffice_location',
            'open_backoffice_location',
            'get_product_info',
            'update_product_field',
            'update_field'
        ];

        if (!in_array($tool, $allowed, true)) {
            throw new \RuntimeException('Tool not allowed: ' . $tool);
        }
    }

    private function log(string $event, array $payload = []): void
    {
        Db::getInstance()->insert('iassistant_log', [
            'event' => pSQL($event),
            'payload' => pSQL(json_encode($payload)),
            'created_at' => date('Y-m-d H:i:s'),
        ]);
    }
}
