<?php
/**
 * iassistant.php — Module principal
 * - Injecte une fenêtre de chat (iframe) partout dans le Back-Office
 * - Signe un jeton HMAC pour authentifier l’employé auprès de l’API
 * - Expose un endpoint interne /api (contrôleur séparé) pour exécuter des tools MCP
 *
 * IMPORTANT :
 *   - L’API et l’URL du chat sont FIXES (non modifiables dans le BO)
 *   - Le secret HMAC est généré à l’installation (stocké en Configuration)
 */

if (!defined('_PS_VERSION_')) { exit; }

class Iassistant extends Module
{
    /*** >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
     *  Remplace UNIQUEMENT ici si le domaine change.
     *  Les marchands ne peuvent pas modifier ces valeurs dans le BO.
     *  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/
    private const API_BASE        = 'https://api.iassistant.shop';
    private const CHAT_IFRAME_URL = 'https://api.iassistant.shop/chat';

    public function __construct()
    {
        $this->name = 'iassistant';
        $this->tab = 'administration';
        $this->version = '1.0.0';
        $this->author = 'Cardinale Ludovic';
        $this->need_instance = 0;
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('iAssistant (LLM + MCP)');
        $this->description = $this->l('Chat Back-Office connecté à votre API iAssistant (Stripe + LLM) et outils MCP PrestaShop.');
    }

    public function install()
    {
        $ok = parent::install()
            && $this->registerHook('displayBackOfficeFooter')     // injection iframe
            && $this->registerHook('actionAdminControllerSetMedia') // style minimal
            && $this->installDb()
            && Configuration::updateValue('IASSISTANT_ENABLED', 1);

        // Génère le secret HMAC si absent (non modifiable via BO)
        if ($ok && !Configuration::get('IASSISTANT_SHARED_SECRET')) {
            Configuration::updateValue('IASSISTANT_SHARED_SECRET', bin2hex(random_bytes(16)));
        }

        // Enregistrer cette instance auprès de l’API (facultatif, non bloquant)
        if ($ok) {
            try { $this->registerInstanceToApi(); } catch (\Throwable $e) { /* ignore soft */ }
        }

        return $ok;
    }

    public function uninstall()
    {
        return Configuration::deleteByName('IASSISTANT_SHARED_SECRET')
            && Configuration::deleteByName('IASSISTANT_ENABLED')
            && parent::uninstall();
    }

    private function installDb(): bool
    {
        $sql = 'CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'iassistant_log` (
            `id_log` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            `event` VARCHAR(64) NOT NULL,
            `payload` LONGTEXT NULL,
            `created_at` DATETIME NOT NULL
        ) ENGINE='._MYSQL_ENGINE_.' DEFAULT CHARSET=utf8mb4;';
        return (bool)Db::getInstance()->execute($sql);
    }

    /**
     * Enregistre la boutique auprès de l’API (optionnel)
     * - envoie : platform, module, version, shop_url, endpoint_url, shared_secret
     * - l’API peut vérifier, lier Stripe, etc.
     */
    private function registerInstanceToApi(): void
    {
        $shopUrl     = Tools::getShopDomainSsl(true).__PS_BASE_URI__;
        $secret      = Configuration::get('IASSISTANT_SHARED_SECRET');
        $endpointUrl = $this->context->link->getModuleLink($this->name, 'api', [], true);

        $payload = [
            'platform'      => 'prestashop',
            'module'        => $this->name,
            'version'       => $this->version,
            'shop_url'      => $shopUrl,
            'endpoint_url'  => $endpointUrl,
            'shared_secret' => $secret,
        ];

        $url = rtrim(self::API_BASE, '/').'/api/integrations/register';
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_HTTPHEADER     => ['Content-Type: application/json'],
            CURLOPT_POSTFIELDS     => json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE),
            CURLOPT_TIMEOUT        => 300,
        ]);
        $resp = curl_exec($ch);
        curl_close($ch);
        // Pas bloquant si échec : l’iframe pourra proposer une étape d’activation côté API.
    }

    /**
     * Page de configuration : lecture seule (les URLs API sont fixées dans le code).
     */
    public function getContent()
    {
        $enabled  = (int)Configuration::get('IASSISTANT_ENABLED');
        $secret   = Configuration::get('IASSISTANT_SHARED_SECRET');
        $endpoint = $this->context->link->getModuleLink($this->name, 'api', [], true);

        $employee = $this->context->employee;
        if (!$employee || !$employee->id) { return ''; }

        // Token HMAC
        $shopUrl = Tools::getShopDomainSsl(true).__PS_BASE_URI__;
        $payload = [
            'email'       => (string) \Configuration::get('PS_SHOP_EMAIL'),
            'shop_url'    => (string) $shopUrl,
            'ts'          => time(),
        ];
        $json   = json_encode($payload);
        $token  = base64_encode($json).'.'.hash_hmac('sha256', $json, $secret);

        $html = '
            <div class="panel">
                <a href="' . self::API_BASE . '/api/billing/portal?shopUrl='.rawurlencode($shopUrl).'&token='.rawurlencode($token).'" target="_blank">Gérer mon abonnement</a>
            </div>
        ';

        return $html;
    }

    /**
     * Style minimal pour la fenêtre de chat (fixe, responsive).
     */
    public function hookActionAdminControllerSetMedia()
    {
        if (!Configuration::get('IASSISTANT_ENABLED')) { return; }
        $this->context->controller->addJS($this->_path.'views/js/iassistant-parent.js');
        $this->context->controller->addCSS($this->_path.'views/css/iassistant-parent.css');
    }

    /**
     * Injection de l’iframe de chat dans tout le Back-Office.
     * Le token passé dans l’URL est signé (HMAC) et contient l’email employé + shop + timestamp.
     */
    public function hookDisplayBackOfficeFooter()
    {
        if (!Configuration::get('IASSISTANT_ENABLED')) { return ''; }

        $employee = $this->context->employee;
        if (!$employee || !$employee->id) { return ''; }

        // Token HMAC
        $shopUrl = Tools::getShopDomainSsl(true).__PS_BASE_URI__;
        $payload = [
            'email'    => (string) Configuration::get('PS_SHOP_EMAIL'),
            'shop_url' => (string) $shopUrl,
            'ts'       => time(),
        ];
        $secret = Configuration::get('IASSISTANT_SHARED_SECRET');
        $json   = json_encode($payload);
        $token  = base64_encode($json).'.'.hash_hmac('sha256', $json, $secret);

        // URL iframe
        $src = rtrim(self::CHAT_IFRAME_URL, '/')
            .'?shopUrl='.rawurlencode($shopUrl)
            .'&employeeId='.(int)$employee->id
            .'&token='.rawurlencode($token)
            .'&module=iassistant'
            .'&v='.rawurlencode($this->version);

        $returnUrl = $this->context->link->getAdminLink('AdminModules', true, [], [
            'configure'   => $this->name,
            'tab_module'  => $this->tab,
            'module_name' => $this->name,
        ]);

        $this->context->smarty->assign([
            'ias_chat_src'   => $src,
            'ias_logo_url'   => $this->_path.'views/img/logo.png',
            'ias_iframe_id'  => 'iassistant-chat',
            'ias_toggle_id'  => 'iassistant-toggle',
            'ias_wrap_id'    => 'iassistant-wrap',
            'current_url_with_token' => $returnUrl
        ]);

        // rend le template Smarty
        return $this->display(__FILE__, 'views/templates/hook/backoffice_footer.tpl');
    }

}
