<?php
namespace Iassistant\Mcp;

use Context;
use Image;
use ImageManager;
use ImageType;
use Product;
use Category;
use Db;
use Configuration;
use StockAvailable;
use Shop;
use Language;
use Validate;
use Tools as PsTools;

class McpTools
{
    private function makeLinkRewrite(string $name): string
    {
        // ✅ Utilise bien la classe PrestaShop globale \Tools
        if (method_exists(\Tools::class, 'link_rewrite')) {
            return (string) \Tools::link_rewrite($name);
        }
        if (method_exists(\Tools::class, 'str2url')) {
            return (string) \Tools::str2url($name);
        }
        // Fallback simple si jamais…
        $slug = @iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $name);
        $slug = strtolower(preg_replace('/[^a-z0-9]+/i', '-', (string) $slug));
        $slug = trim($slug, '-');
        return $slug !== '' ? $slug : 'product';
    }

    private function getProductLinkRewrite(Product $product, int $idLang): string
    {
        $lr = $product->link_rewrite ?? '';
        if (is_array($lr)) {
            $val = $lr[$idLang] ?? reset($lr);
            if (is_string($val) && $val !== '') { return $val; }
        } elseif (is_string($lr) && $lr !== '') {
            return $lr;
        }
        $name = is_array($product->name) ? ($product->name[$idLang] ?? reset($product->name)) : (string)$product->name;
        return $this->makeLinkRewrite($name) ?: '';
    }

    private function convertToJpegBinary(string $binary, string $mime): string
    {
        if (stripos($mime, 'jpeg') !== false || stripos($mime, 'jpg') !== false) {
            return $binary;
        }
        if (!function_exists('imagecreatefromstring')) {
            throw new \RuntimeException('GD extension required to convert images.');
        }
        $src = @imagecreatefromstring($binary);
        if (!$src) { throw new \RuntimeException('Cannot decode image'); }
        $w = imagesx($src); $h = imagesy($src);
        $dst = imagecreatetruecolor($w, $h);
        $white = imagecolorallocate($dst, 255, 255, 255);
        imagefill($dst, 0, 0, $white);
        imagecopy($dst, $src, 0, 0, 0, 0, $w, $h);
        ob_start(); imagejpeg($dst, null, 92); $jpeg = (string)ob_get_clean();
        imagedestroy($src); imagedestroy($dst);
        if ($jpeg === '') { throw new \RuntimeException('JPEG conversion failed'); }
        return $jpeg;
    }

    /** 1) Lister les images d’un produit */
    public function listProductImages(int $productId, int $langId, string $imageType = 'large_default'): array
    {
        $product = new Product($productId, true, $langId);
        if (!\Validate::isLoadedObject($product)) {
            throw new \RuntimeException('Product not found: '.$productId);
        }
        $linkRewrite = $this->getProductLinkRewrite($product, $langId);
        $rows = Image::getImages($langId, $productId);

        $out = [];
        foreach ($rows as $row) {
            $idImage = (int)$row['id_image'];
            $out[] = [
                'id_image' => $idImage,
                'position' => isset($row['position']) ? (int)$row['position'] : 0,
                'cover' => !empty($row['cover']),
                'url' => Context::getContext()->link->getImageLink($linkRewrite, $idImage, $imageType),
            ];
        }
        return $out;
    }

    /** 2) Obtenir une image en base64 */
    public function getProductImageBase64(int $imageId): array
    {
        $image = new Image($imageId);
        $original = 'img/p/'.$image->getExistingImgPath().'.jpg';
        if (!file_exists($original)) {
            throw new \RuntimeException('Original not found: '.$original);
        }
        $bin = (string)file_get_contents($original);
        return ['mime'=>'image/jpeg', 'image_b64'=>base64_encode($bin), 'id_image'=>(int)$imageId];
    }

    /** 3) Enregistrer une image générée (base64) */
    public function saveGeneratedImage(int $productId, string $imageB64, bool $setAsCover = false): array
    {
        $product = new Product($productId);
        if (!\Validate::isLoadedObject($product)) {
            throw new \RuntimeException('Product not found: '.$productId);
        }
        $bin = base64_decode($imageB64, true);
        if ($bin === false) { throw new \RuntimeException('Invalid base64'); }

        $mime = 'image/jpeg';
        if (function_exists('finfo_buffer')) {
            $f = new \finfo(FILEINFO_MIME_TYPE);
            $det = $f->buffer($bin);
            if (is_string($det) && $det) { $mime = $det; }
        }
        if ($mime !== 'image/jpeg' && $mime !== 'image/jpg') {
            $bin = $this->convertToJpegBinary($bin, $mime);
            $mime = 'image/jpeg';
        }

        $image = new Image();
        $image->id_product = (int)$productId;
        $image->position   = Image::getHighestPosition($productId) + 1;
        $image->cover      = (int)$setAsCover;
        if (!$image->add()) { throw new \RuntimeException('Failed to create Image row'); }

        $basePath = $image->getPathForCreation();
        $origPath = $basePath.'.jpg';
        if (!is_dir(dirname($origPath))) { @mkdir(dirname($origPath), 0775, true); }
        if (file_put_contents($origPath, $bin) === false) {
            $image->delete();
            throw new \RuntimeException('Cannot write original image');
        }

        $types = ImageType::getImagesTypes('products');
        foreach ($types as $type) {
            $dest = $basePath.'-'.$type['name'].'.jpg';
            ImageManager::resize($origPath, $dest, (int)$type['width'], (int)$type['height'], 'jpg', true);
        }

        if ($setAsCover) {
            Image::deleteCover($productId);
            $image->cover = 1;
            $image->update();
        }

        $langId = (int)Context::getContext()->language->id;
        $linkRewrite = $this->getProductLinkRewrite($product, $langId);
        $url = Context::getContext()->link->getImageLink($linkRewrite, (int)$image->id, 'large_default');

        return ['id_image'=>(int)$image->id, 'cover'=>(bool)$image->cover, 'url'=>$url];
    }

    /** 4) Régénérer les miniatures d’une image */
    public function regenerateProductImageThumbnails(int $imageId, bool $cleanExisting = false): array
    {
        $image = new Image($imageId);
        if (!\Validate::isLoadedObject($image)) {
            throw new \RuntimeException('Image not found: '.$imageId);
        }
        $existingBase = 'img/p/'.$image->getExistingImgPath();
        $basePath     = $image->getPathForCreation();
        $originalJpg  = $existingBase.'.jpg';
        if (!file_exists($originalJpg)) {
            throw new \RuntimeException('Original not found: '.$originalJpg);
        }

        $types = ImageType::getImagesTypes('products');
        $generated = 0; $details = [];
        foreach ($types as $type) {
            $name = (string)$type['name']; $w=(int)$type['width']; $h=(int)$type['height'];
            $dest = $basePath.'-'.$name.'.jpg';
            if ($cleanExisting && file_exists($dest)) { @unlink($dest); }
            $ok = ImageManager::resize($originalJpg, $dest, $w, $h, 'jpg', true);
            if ($ok) { $generated++; }
            $details[] = ['name'=>$name,'width'=>$w,'height'=>$h,'path'=>$dest];
        }
        return ['imageId'=>(int)$image->id,'generated'=>$generated,'types'=>$details];
    }

    /** 5) Définir une image comme cover */
    public function setProductCoverImage(int $productId, int $imageId): array
    {
        $product = new Product($productId);
        if (!\Validate::isLoadedObject($product)) {
            throw new \RuntimeException('Product not found: '.$productId);
        }
        $image = new Image($imageId);
        if (!\Validate::isLoadedObject($image) || (int)$image->id_product !== (int)$productId) {
            throw new \RuntimeException('Image not found for this product');
        }

        if (\Shop::isFeatureActive() && method_exists($image, 'associateTo')) {
            $shopId = (int)Context::getContext()->shop->id;
            $image->associateTo([$shopId]);
        }

        Image::deleteCover($productId);
        $image->cover = 1;
        if (!$image->update()) { throw new \RuntimeException('Failed to set cover'); }

        $langId = (int)Context::getContext()->language->id;
        $linkRewrite = $this->getProductLinkRewrite($product, $langId);
        $url = Context::getContext()->link->getImageLink($linkRewrite, $imageId, 'large_default');

        return ['success'=>true,'id_image'=>(int)$imageId,'cover'=>true,'url'=>$url];
    }

    /**
     * 6) Générer une nouvelle image depuis une image existante (via API iassistant)
     * Appelle l’API: /api/mcp/generate-product-image (Gemini + sauvegarde + miniatures côté API),
     * puis renvoie le résultat (ex: id_image, url…).
     */
    public function generateNewImageFromExisting(
        int $productId,
        int $imageId,
        string $prompt,
        bool $setAsCover = false
    ): array {
        $apiBase = (string) \Configuration::get('IASSISTANT_API_BASE');
        if (!$apiBase) {
            $apiBase = 'https://api.iassistant.shop';
        }
        $url = rtrim($apiBase, '/') . '/api/mcp/generate-product-image';

        $token = $this->buildApiToken();

        $payload = [
            'token'      => $token,
            'productId'  => $productId,
            'imageId'    => $imageId,
            'prompt'     => $prompt,
            'setAsCover' => (bool) $setAsCover,
        ];

        $out = $this->postJson($url, $payload);

        if (!isset($out['ok']) || $out['ok'] !== true) {
            $msg = isset($out['error']) ? (string) $out['error'] : 'unknown_error';
            throw new \RuntimeException('generation_failed: ' . $msg);
        }

        // $out['result'] contient typiquement: id_image, cover, url…
        return (array) $out['result'];
    }

    /**
     * Construit un token HMAC pour l’API iassistant (même format que l’iframe).
     */
    private function buildApiToken(): string
    {
        $secret = (string) \Configuration::get('IASSISTANT_SHARED_SECRET');
        if (!$secret) {
            throw new \RuntimeException('missing_shared_secret');
        }

        $ctx = \Context::getContext();
        $employee = $ctx && $ctx->employee ? $ctx->employee : null;

        $shopUrl = \Tools::getShopDomainSsl(true) . __PS_BASE_URI__;

        $payload = [
            'email'       => (string) \Configuration::get('PS_SHOP_EMAIL'),
            'shop_url'    => (string) $shopUrl,
            'ts'          => time(),
        ];

        $json = json_encode($payload, JSON_UNESCAPED_SLASHES);
        if ($json === false) {
            throw new \RuntimeException('token_json_encode_failed');
        }

        $sig = hash_hmac('sha256', $json, $secret);
        return base64_encode($json) . '.' . $sig;
    }

    /**
     * POST JSON simple (cURL) avec timeouts.
     */
    private function postJson(string $url, array $payload): array
    {
        $ch = curl_init($url);
        if (!$ch) {
            throw new \RuntimeException('curl_init_failed');
        }

        $body = json_encode($payload, JSON_UNESCAPED_SLASHES);
        if ($body === false) {
            throw new \RuntimeException('payload_json_encode_failed');
        }

        curl_setopt_array($ch, [
            CURLOPT_POST            => true,
            CURLOPT_RETURNTRANSFER  => true,
            CURLOPT_HTTPHEADER      => ['Content-Type: application/json'],
            CURLOPT_TIMEOUT         => 300,
            CURLOPT_CONNECTTIMEOUT  => 10,
            CURLOPT_POSTFIELDS      => $body,
            CURLOPT_SSL_VERIFYHOST  => 2,
            CURLOPT_SSL_VERIFYPEER  => true,
        ]);

        $resp = curl_exec($ch);
        $errno = curl_errno($ch);
        $err   = curl_error($ch);
        $code  = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($errno) {
            throw new \RuntimeException('curl_error: ' . $err);
        }
        if ($code < 200 || $code >= 300) {
            throw new \RuntimeException('http_error_' . $code . ': ' . (string) $resp);
        }

        $json = json_decode((string) $resp, true);
        if (!is_array($json)) {
            throw new \RuntimeException('invalid_json_response');
        }
        return $json;
    }

    /** 6) Résoudre un langId depuis isoCode (fr, en) ou locale (fr-FR).
     *    Si rien n’est fourni, renvoie la langue du contexte.
     *    Retour: ['langId'=>int, 'iso_code'=>string, 'locale'=>string, 'resolvedBy'=>string]
     */
    public function getLangId(?string $isoCode = null, ?string $locale = null): array
    {
        $isoCode = $isoCode !== null ? trim(strtolower($isoCode)) : null;
        $locale  = $locale  !== null ? trim(strtolower($locale))  : null;

        // 1) Si isoCode fourni -> méthode native
        if ($isoCode) {
            $id = (int)\Language::getIdByIso($isoCode);
            if ($id > 0) {
                $row = $this->findLangRowById($id);
                return [
                    'langId'    => $id,
                    'iso_code'  => strtolower((string)$row['iso_code']),
                    'locale'    => strtolower((string)($row['locale'] ?? $row['language_code'] ?? '')),
                    'resolvedBy'=> 'iso',
                ];
            }
        }

        // 2) Si locale fourni -> boucle sur les langues
        if ($locale) {
            foreach (\Language::getLanguages(false) as $l) { // false = inclut inactives si besoin
                $lc = strtolower((string)($l['locale'] ?? $l['language_code'] ?? ''));
                if ($lc && $lc === $locale) {
                    return [
                        'langId'    => (int)$l['id_lang'],
                        'iso_code'  => strtolower((string)$l['iso_code']),
                        'locale'    => $lc,
                        'resolvedBy'=> 'locale',
                    ];
                }
            }
        }

        // 3) Sinon -> langue du contexte, puis défaut boutique
        $ctx = \Context::getContext();
        $ctxId = (int)($ctx && $ctx->language ? $ctx->language->id : 0);
        if ($ctxId > 0) {
            $row = $this->findLangRowById($ctxId);
            return [
                'langId'    => $ctxId,
                'iso_code'  => strtolower((string)$row['iso_code']),
                'locale'    => strtolower((string)($row['locale'] ?? $row['language_code'] ?? '')),
                'resolvedBy'=> 'context',
            ];
        }

        $defId = (int)\Configuration::get('PS_LANG_DEFAULT');
        if ($defId > 0) {
            $row = $this->findLangRowById($defId);
            return [
                'langId'    => $defId,
                'iso_code'  => strtolower((string)$row['iso_code']),
                'locale'    => strtolower((string)($row['locale'] ?? $row['language_code'] ?? '')),
                'resolvedBy'=> 'default',
            ];
        }

        throw new \RuntimeException('No language found');
    }

    /** Helper: retrouve la ligne langue par id_lang */
    private function findLangRowById(int $idLang): array
    {
        foreach (\Language::getLanguages(false) as $l) {
            if ((int)$l['id_lang'] === $idLang) return $l;
        }
        // fallback: recharger un objet Language
        $lang = new \Language($idLang);
        if (\Validate::isLoadedObject($lang)) {
            return [
                'id_lang' => (int)$lang->id,
                'iso_code'=> (string)$lang->iso_code,
                'locale'  => (string)($lang->locale ?: $lang->language_code),
            ];
        }
        return ['id_lang'=>0,'iso_code'=>'','locale'=>''];
    }

    public function listLanguages(bool $activeOnly = true): array
    {
        // true => seulement actives, false => toutes
        $langs = \Language::getLanguages($activeOnly);
        $out = [];
        foreach ($langs as $l) {
            $out[] = [
                'langId'    => (int)$l['id_lang'],
                'iso_code'  => strtolower((string)$l['iso_code']),
                'locale'    => strtolower((string)($l['locale'] ?? $l['language_code'] ?? '')),
                'name'      => (string)($l['name'] ?? ''),
                'active'    => isset($l['active']) ? (bool)$l['active'] : true,
            ];
        }
        return $out;
    }

    /** 7) Créer un produit (champs minimum) */
    public function createProduct(array $data): array
    {
        $langId = (int)($data['langId'] ?? (int)Context::getContext()->language->id);
        $name = (string)$data['name'];
        $description = (string)($data['description'] ?? '');
        $description_short = (string)($data['description_short'] ?? '');
        $price = (float)($data['price'] ?? 0.0);
        $meta_title = (string)($data['meta_title'] ?? '');
        $meta_description = (string)($data['meta_description'] ?? '');
        $categoryId = isset($data['categoryId']) ? (int)$data['categoryId'] : (int)\Configuration::get('PS_HOME_CATEGORY');
        $active = isset($data['active']) ? (bool)$data['active'] : true;

        if ($name === '') { throw new \RuntimeException('Missing name'); }

        $p = new \Product();

        // Champs multi-langues
        $p->name = [$langId => $name];
        $p->link_rewrite = [$langId => $this->makeLinkRewrite($name)];
        $p->description = [$langId => $description];
        $p->description_short = [$langId => $description_short];
        $p->meta_title = [$langId => $meta_title];
        $p->meta_description = [$langId => $meta_description];

        $p->price = $price; // HT
        $p->id_category_default = $categoryId;
        $p->active = (int)$active;

        if (!$p->add()) {
            throw new \RuntimeException('Failed to create product');
        }

        // Associer catégorie
        if (method_exists($p, 'addToCategories')) {
            $p->addToCategories([$categoryId]);
        }

        return ['ok'=>true, 'id_product'=>(int)$p->id];
    }

    /** 8) Ajouter plusieurs images base64 d’un coup */
    public function addProductImagesBase64(int $productId, array $imagesB64, bool $setFirstAsCover = true): array
    {
        $product = new \Product($productId);
        if (!\Validate::isLoadedObject($product)) {
            throw new \RuntimeException('Product not found: '.$productId);
        }

        $saved = [];
        foreach (array_values($imagesB64) as $i => $b64) {
            $bin = base64_decode($b64, true);
            if ($bin === false) { continue; }

            // détecter le mime
            $mime = 'image/jpeg';
            if (function_exists('finfo_buffer')) {
                $f = new \finfo(FILEINFO_MIME_TYPE);
                $det = $f->buffer($bin);
                if (is_string($det) && $det) { $mime = $det; }
            }
            if ($mime !== 'image/jpeg' && $mime !== 'image/jpg') {
                $bin = $this->convertToJpegBinary($bin, $mime);
                $mime = 'image/jpeg';
            }

            $image = new \Image();
            $image->id_product = (int)$productId;
            $image->position   = \Image::getHighestPosition($productId) + 1;
            $image->cover      = (int)($setFirstAsCover && $i === 0);
            if (!$image->add()) { continue; }

            $basePath = $image->getPathForCreation();
            $origPath = $basePath.'.jpg';
            if (!is_dir(dirname($origPath))) { @mkdir(dirname($origPath), 0775, true); }
            if (file_put_contents($origPath, $bin) === false) {
                $image->delete();
                continue;
            }

            $types = \ImageType::getImagesTypes('products');
            foreach ($types as $type) {
                $dest = $basePath.'-'.$type['name'].'.jpg';
                \ImageManager::resize($origPath, $dest, (int)$type['width'], (int)$type['height'], 'jpg', true);
            }

            if ($setFirstAsCover && $i === 0) {
                \Image::deleteCover($productId);
                $image->cover = 1;
                $image->update();
            }

            $langId = (int)\Context::getContext()->language->id;
            $linkRewrite = $this->getProductLinkRewrite($product, $langId);
            $url = \Context::getContext()->link->getImageLink($linkRewrite, (int)$image->id, 'large_default');
            $saved[] = ['id_image'=>(int)$image->id, 'url'=>$url, 'cover'=>(bool)$image->cover];
        }

        return ['ok'=>true, 'count'=>count($saved), 'images'=>$saved];
    }
 
    /**
     * Liste les produits qui se vendent le moins sur une période, avec un diagnostic et des recommandations.
     *
     * @param array $args [
     *   'date_from' => '2025-08-01', // inclusif
     *   'date_to'   => '2025-09-01', // exclusif
     *   'limit'     => 20,
     *   'min_orders'=> 3,            // nombre de ventes max pour considérer "faible"
     *   'id_lang'   => 1,
     *   'id_shop'   => 1,
     *   'exclude_oos' => false       // exclure les produits hors stock des résultats
     * ]
     * @return array
     */
    public function find_low_selling_products(array $args)
    {
        $ctx = Context::getContext();

        $idShop    = (int)($args['id_shop'] ?? ($ctx->shop ? (int)$ctx->shop->id : (int)Shop::getContextShopID()));
        $idLang    = (int)($args['id_lang'] ?? ($ctx->language ? (int)$ctx->language->id : (int)Configuration::get('PS_LANG_DEFAULT')));
        $dateFrom  = pSQL($args['date_from'] ?? date('Y-m-01'));
        $dateTo    = pSQL($args['date_to']   ?? date('Y-m-d'));
        $limit     = (int)($args['limit'] ?? 20);
        $minOrders = (int)($args['min_orders'] ?? 3);
        $excludeOos = !empty($args['exclude_oos']);

        // Etats considérés comme "payés/valides" (adapter selon ta boutique)
        $paidStates = $args['paid_states'] ?? [2, 3, 4]; // ex: 2=Payment accepted, etc.
        $paidStates = array_map('intval', (array)$paidStates);
        if (!$paidStates) { $paidStates = [2]; }
        $paidStatesList = implode(',', $paidStates);

        $db = Db::getInstance(_PS_USE_SQL_SLAVE_);

        // 1) Ventes par produit sur la période (date_to inclus)
        $sqlSales = '
            SELECT
                od.product_id AS id_product,
                SUM(od.product_quantity)       AS sales_qty,
                SUM(od.total_price_tax_incl)   AS revenue
            FROM '._DB_PREFIX_.'order_detail od
            INNER JOIN '._DB_PREFIX_.'orders o ON o.id_order = od.id_order
            WHERE o.id_shop = '.(int)$idShop.'
            AND o.date_add >= "'.$dateFrom.'"
            AND o.date_add < DATE_ADD("'.$dateTo.'", INTERVAL 1 DAY)
            AND o.current_state IN ('.$paidStatesList.')
            GROUP BY od.product_id
            HAVING sales_qty <= '.(int)$minOrders.'
            ORDER BY sales_qty ASC, revenue ASC
            LIMIT '.(int)$limit;

        $sales = $db->executeS($sqlSales) ?: [];

        // Fallback: produits actifs (zéro vente trouvée)
        if (empty($sales)) {
            $sqlZero = '
                SELECT p.id_product, 0 AS sales_qty, 0 AS revenue
                FROM '._DB_PREFIX_.'product p
                INNER JOIN '._DB_PREFIX_.'product_shop ps ON ps.id_product = p.id_product AND ps.id_shop = '.(int)$idShop.'
                WHERE ps.active = 1
                ORDER BY p.id_product DESC
                LIMIT '.(int)$limit;
            $sales = $db->executeS($sqlZero) ?: [];
        }

        if (empty($sales)) {
            return ['items' => [], 'meta' => ['count' => 0, 'note' => 'Aucun produit trouvé pour les critères.']];
        }

        $ids = array_map(static function($r){ return (int)$r['id_product']; }, $sales);
        $idList = implode(',', array_map('intval', $ids));

        // 2) Métadonnées produits (+ join shop s pour id_shop_group)
        $sqlMeta = '
            SELECT
                p.id_product,
                pl.name,
                pl.link_rewrite,
                ps.active,
                ps.visibility,
                ps.price,
                pl.meta_title,
                pl.meta_description,
                pl.description_short,
                p.id_category_default AS id_category,
                cl.name AS category_name,
                (
                    SELECT COUNT(*)
                    FROM '._DB_PREFIX_.'image i
                    INNER JOIN '._DB_PREFIX_.'image_shop ish
                        ON ish.id_image = i.id_image AND ish.id_shop = ps.id_shop
                    WHERE i.id_product = p.id_product
                ) AS image_count,
                (
                    SELECT COUNT(*)
                    FROM '._DB_PREFIX_.'image i2
                    INNER JOIN '._DB_PREFIX_.'image_shop ish2
                        ON ish2.id_image = i2.id_image AND ish2.id_shop = ps.id_shop
                    WHERE i2.id_product = p.id_product AND ish2.cover = 1
                ) AS has_cover,
                (
                    SELECT sa.quantity
                    FROM '._DB_PREFIX_.'stock_available sa
                    WHERE sa.id_product = p.id_product
                    AND sa.id_product_attribute = 0
                    AND (
                            sa.id_shop = ps.id_shop
                        OR (sa.id_shop = 0 AND sa.id_shop_group = s.id_shop_group)
                        OR (sa.id_shop IS NULL AND sa.id_shop_group = s.id_shop_group)
                    )
                    ORDER BY sa.id_shop DESC
                    LIMIT 1
                ) AS stock_qty,
                (
                    SELECT COUNT(*)
                    FROM '._DB_PREFIX_.'product_tag pt
                    WHERE pt.id_product = p.id_product
                ) AS tag_count,
                (
                    SELECT COUNT(*)
                    FROM '._DB_PREFIX_.'product_attribute pa
                    WHERE pa.id_product = p.id_product
                ) AS has_attributes
            FROM '._DB_PREFIX_.'product p
            INNER JOIN '._DB_PREFIX_.'product_shop ps
                ON ps.id_product = p.id_product AND ps.id_shop = '.(int)$idShop.'
            INNER JOIN '._DB_PREFIX_.'shop s ON s.id_shop = ps.id_shop
            INNER JOIN '._DB_PREFIX_.'product_lang pl
                ON pl.id_product = p.id_product AND pl.id_lang = '.(int)$idLang.' AND pl.id_shop = ps.id_shop
            LEFT JOIN '._DB_PREFIX_.'category_lang cl
                ON cl.id_category = p.id_category_default AND cl.id_lang = '.(int)$idLang.'
            WHERE p.id_product IN ('.$idList.')
        ';
        $metaRows = $db->executeS($sqlMeta) ?: [];
        $metaById = [];
        foreach ($metaRows as $r) {
            $metaById[(int)$r['id_product']] = $r;
        }

        // 3) Référence de prix par catégorie (AVG, compatible MySQL)
        $catIds = array_values(array_unique(array_filter(array_map(static function($r){ return (int)$r['id_category']; }, $metaRows))));
        $catMedian = [];
        if ($catIds) {
            foreach ($catIds as $idCat) {
                $sqlAvg = '
                    SELECT AVG(ps2.price) AS avg_price
                    FROM '._DB_PREFIX_.'product_shop ps2
                    INNER JOIN '._DB_PREFIX_.'category_product cp2 ON cp2.id_product = ps2.id_product
                    WHERE ps2.id_shop = '.(int)$idShop.' AND cp2.id_category = '.(int)$idCat;
                $avg = (float)$db->getValue($sqlAvg);
                $catMedian[(int)$idCat] = $avg; // "median" approx par moyenne
            }
        }

        // 4) Construire diagnostic
        $items = [];
        foreach ($sales as $row) {
            $idProduct = (int)$row['id_product'];
            if (!isset($metaById[$idProduct])) {
                continue;
            }
            $m = $metaById[$idProduct];

            $diagnosis = [];
            $reco = [];

            $salesQty = (int)$row['sales_qty'];
            $revenue  = round((float)$row['revenue'], 2);
            $stock    = (int)$m['stock_qty'];
            $hasCover = ((int)$m['has_cover']) > 0;
            $imgCount = (int)$m['image_count'];
            $price    = (float)$m['price'];
            $idCat    = (int)$m['id_category'];
            $catName  = (string)$m['category_name'];
            $medianCat = isset($catMedian[$idCat]) ? (float)$catMedian[$idCat] : null;

            if ($excludeOos && $stock <= 0) {
                continue;
            }

            if (!$m['active']) {
                $diagnosis[] = 'Produit inactif.';
                $reco[] = ['action' => 'set_active', 'params' => ['active' => true]];
            }

            if (!$hasCover) {
                $diagnosis[] = 'Aucune image de couverture.';
                $reco[] = ['action' => 'set_cover_from_first', 'params' => []];
            }

            if ($imgCount < 3) {
                $diagnosis[] = 'Peu d’images ('.$imgCount.').';
                $reco[] = ['action' => 'request_new_images', 'params' => ['min_images' => 3]];
            }

            $mt = trim((string)$m['meta_title']);
            $md = trim((string)$m['meta_description']);
            if ($mt === '' || mb_strlen($mt) < 25) {
                $diagnosis[] = 'Meta title faible/absent.';
                $reco[] = ['action' => 'improve_meta_title', 'params' => []];
            }
            if ($md === '' || mb_strlen($md) < 50) {
                $diagnosis[] = 'Meta description faible/absente.';
                $reco[] = ['action' => 'improve_meta_description', 'params' => []];
            }
            $ds = trim(strip_tags((string)$m['description_short']));
            if ($ds === '' || mb_strlen($ds) < 60) {
                $diagnosis[] = 'Description courte trop courte.';
                $reco[] = ['action' => 'improve_short_description', 'params' => []];
            }

            if ($medianCat && $medianCat > 0) {
                if ($price > 1.25 * $medianCat) {
                    $diagnosis[] = 'Prix possiblement élevé vs catégorie.';
                    $reco[] = ['action' => 'add_discount', 'params' => ['type' => 'percent', 'value' => 10, 'duration_days' => 14]];
                } elseif ($price < 0.7 * $medianCat) {
                    $diagnosis[] = 'Prix possiblement trop bas (dévalorisation).';
                    $reco[] = ['action' => 'review_price_positioning', 'params' => []];
                }
            }

            if ($stock <= 0) {
                $diagnosis[] = 'Rupture de stock.';
                $reco[] = ['action' => 'restock', 'params' => ['target' => 20]];
            } elseif ($stock < 3) {
                $diagnosis[] = 'Stock très bas ('.$stock.').';
                $reco[] = ['action' => 'restock', 'params' => ['target' => 20]];
            }

            if ((int)$m['tag_count'] === 0) {
                $diagnosis[] = 'Aucun tag.';
                $reco[] = ['action' => 'add_tags', 'params' => ['tags' => []]];
            }

            if ((int)$m['has_attributes'] > 0) {
                $diagnosis[] = 'Vérifier les images de variantes.';
                $reco[] = ['action' => 'ensure_variant_images', 'params' => []];
            }

            $items[] = [
                'id_product'      => $idProduct,
                'name'            => $m['name'],
                'category_name'   => $catName,
                'sales_qty'       => $salesQty,
                'revenue'         => $revenue,
                'stock'           => $stock,
                'price'           => $price,
                'has_cover'       => (bool)$hasCover,
                'image_count'     => $imgCount,
                'active'          => (bool)$m['active'],
                'diagnosis'       => $diagnosis,
                'recommendations' => $reco,
            ];
        }

        return [
            'items' => $items,
            'meta' => [
                'count'  => count($items),
                'period' => ['from' => $dateFrom, 'to' => $dateTo],
                'shop'   => $idShop,
                'lang'   => $idLang,
            ],
        ];
    }

    /**
     * Applique une liste d’actions sur un produit, de façon idempotente et sécurisée.
     *
     * @param array $args [
     *   'id_product' => 123,
     *   'id_lang' => 1,
     *   'id_shop' => 1,
     *   'actions' => [
     *       ['action' => 'set_active', 'params' => ['active' => true]],
     *       ['action' => 'add_discount', 'params' => ['type' => 'percent', 'value' => 10, 'duration_days' => 14]],
     *       ...
     *   ]
     * ]
     * @return array
     */
    public function apply_product_recommendations(array $args)
    {
        $ctx      = Context::getContext();

        if (isset($args['productId']) && !isset($args['id_product'])) $args['id_product'] = $args['productId'];
        if (isset($args['langId'])   && !isset($args['id_lang']))    $args['id_lang']    = $args['langId'];
        if (isset($args['shopId'])   && !isset($args['id_shop']))    $args['id_shop']    = $args['shopId'];

        $idShop   = (int)($args['id_shop'] ?? ($ctx->shop ? (int)$ctx->shop->id : (int)Shop::getContextShopID()));
        $idLang   = (int)($args['id_lang'] ?? ($ctx->language ? (int)$ctx->language->id : (int)Configuration::get('PS_LANG_DEFAULT')));
        $idProduct = (int)($args['id_product'] ?? 0);
        $actions   = $args['actions'] ?? [];

        if ($idProduct <= 0 || empty($actions) || !Validate::isUnsignedId($idProduct)) {
            return ['ok' => false, 'error' => 'Invalid id_product or empty actions'];
        }

        $product = new Product($idProduct, false, $idLang, $idShop);
        if (!Validate::isLoadedObject($product)) {
            return ['ok' => false, 'error' => 'Product not found'];
        }

        // Limites "safe" usuelles
        $MAX_META_TITLE        = 255;
        $MAX_META_DESCRIPTION  = 512; // selon schéma PS
        $MAX_DESCRIPTION_SHORT = 900; // soft limit UI

        $results = [];

        foreach ($actions as $idx => $act) {
            $name   = $act['action'] ?? null;
            $params = $act['params'] ?? [];
            $res    = ['action' => $name, 'ok' => false];

            try {
                switch ($name) {

                    case 'set_active': {
                        $active = isset($params['active']) ? (bool)$params['active'] : true;
                        $product->active = $active;
                        // champ product_shop → préciser qu'on le met à jour pour ce shop
                        $product->setFieldsToUpdate(['active' => true]);
                        $res['ok'] = (bool)$product->update();
                        break;
                    }

                    case 'set_price': {
                        $price = (float)($params['price'] ?? -1);
                        if ($price > 0) {
                            // Prix HT de base (les Specific Prices peuvent toujours s'appliquer)
                            $product->price = $price;
                            $product->setFieldsToUpdate(['price' => true]);
                            $res['ok'] = (bool)$product->update();
                        } else {
                            throw new \Exception('Invalid price');
                        }
                        break;
                    }

                    case 'set_cover_from_first': {
                        $images = $product->getImages($idLang);
                        if (!empty($images)) {
                            $firstIdImage = (int)$images[0]['id_image'];
                            $res['ok'] = (bool)$this->setCoverImage($idProduct, $firstIdImage, $idShop);
                        } else {
                            $res['ok']  = false;
                            $res['note'] = 'No images found';
                        }
                        break;
                    }

                    case 'set_cover': {
                        $idImage = (int)($params['id_image'] ?? 0);
                        if ($idImage > 0) {
                            $res['ok'] = (bool)$this->setCoverImage($idProduct, $idImage, $idShop);
                        } else {
                            throw new \Exception('Missing id_image');
                        }
                        break;
                    }

                    case 'improve_meta_title': {
                        if (!empty($params['meta_title'])) {
                            $title = trim((string)$params['meta_title']);
                            if (mb_strlen($title) > $MAX_META_TITLE) {
                                $title = mb_substr($title, 0, $MAX_META_TITLE);
                            }
                            // champs multilingues → tableau id_lang => valeur
                            $product->meta_title = [$idLang => $title];
                            $product->setFieldsToUpdate(['meta_title' => true]);
                            $res['ok'] = (bool)$product->update();
                        } else {
                            $res['ok']   = false;
                            $res['note'] = 'meta_title missing (let AI provide text)';
                        }
                        break;
                    }

                    case 'improve_meta_description': {
                        if (!empty($params['meta_description'])) {
                            $desc = trim((string)$params['meta_description']);
                            if (mb_strlen($desc) > $MAX_META_DESCRIPTION) {
                                $desc = mb_substr($desc, 0, $MAX_META_DESCRIPTION);
                            }
                            $product->meta_description = [$idLang => $desc];
                            $product->setFieldsToUpdate(['meta_description' => true]);
                            $res['ok'] = (bool)$product->update();
                        } else {
                            $res['ok']   = false;
                            $res['note'] = 'meta_description missing (let AI provide text)';
                        }
                        break;
                    }

                    case 'improve_short_description': {
                        if (!empty($params['description_short'])) {
                            $short = trim((string)$params['description_short']);
                            if (mb_strlen($short) > $MAX_DESCRIPTION_SHORT) {
                                $short = mb_substr($short, 0, $MAX_DESCRIPTION_SHORT);
                            }
                            $product->description_short = [$idLang => $short];
                            $product->setFieldsToUpdate(['description_short' => true]);
                            $res['ok'] = (bool)$product->update();
                        } else {
                            $res['ok']   = false;
                            $res['note'] = 'description_short missing (let AI provide text)';
                        }
                        break;
                    }

                    case 'add_tags': {
                        $tags = $params['tags'] ?? [];
                        if (!is_array($tags)) { $tags = []; }
                        // Nettoyage simple (trim, longueur maxi)
                        $tags = array_values(array_filter(array_map(static function ($t) {
                            $t = trim((string)$t);
                            if ($t === '') { return null; }
                            return Tools::substr($t, 0, 32); // coupe à 32 chars par sécurité
                        }, $tags)));

                        $res['ok'] = (bool)$this->addTags($idProduct, $idLang, $tags);
                        break;
                    }

                    case 'add_discount': {
                        $type     = $params['type'] ?? 'percent'; // percent|amount
                        $value    = (float)($params['value'] ?? 0);
                        $duration = (int)($params['duration_days'] ?? 14);
                        $res['ok'] = (bool)$this->createSpecificPrice($product, $type, $value, $duration, $idShop);
                        break;
                    }

                    case 'restock': {
                        // Sécurité : on ne modifie pas le stock physique depuis ici.
                        $res['ok']   = true;
                        $res['note'] = 'Restock requested; please sync with inventory/ERP.';
                        break;
                    }

                    case 'ensure_variant_images': {
                        $res['ok']   = true;
                        $res['note'] = 'Check variant images TODO (report only).';
                        break;
                    }

                    case 'review_price_positioning': {
                        $res['ok']   = true;
                        $res['note'] = 'Price review requested (needs human/AI decision).';
                        break;
                    }

                    case 'regenerate_thumbnails': {
                        // Lancer via un tool séparé (pipeline)
                        $res['ok']   = true;
                        $res['note'] = 'Call regenerate_product_image_thumbnails via MCP.';
                        break;
                    }

                    case 'request_new_images': {
                        $min       = (int)($params['min_images'] ?? 3);
                        $res['ok'] = true;
                        $res['note'] = 'Request new images (min '.$min.') — trigger AI pipeline if available.';
                        break;
                    }

                    default:
                        $res['ok']    = false;
                        $res['error'] = 'Unknown action';
                }

            } catch (\Exception $e) {
                $res['ok']    = false;
                $res['error'] = $e->getMessage();
            }

            $results[] = $res;
        }

        return ['ok' => true, 'results' => $results];
    }

    /**
     * Définit l’image de couverture pour un produit.
     */
    protected function setCoverImage(int $idProduct, int $idImage, int $idShop): bool
    {
        $db = Db::getInstance();
        // reset
        $db->update('image_shop', ['cover' => 0], 'id_image IN (SELECT id_image FROM '._DB_PREFIX_.'image WHERE id_product='.(int)$idProduct.') AND id_shop='.(int)$idShop);
        // set
        return $db->update('image_shop', ['cover' => 1], 'id_image='.(int)$idImage.' AND id_shop='.(int)$idShop);
    }

    /**
     * Ajoute des tags (évite les doublons).
     */
    protected function addTags(int $idProduct, int $idLang, array $tags): bool
    {
        $tags = array_values(array_unique(array_filter(array_map('trim', $tags))));
        if (empty($tags)) {
            return true;
        }

        // PrestaShop fournit Tag::addTags(), mais on fait un insert simple/dédoublonné
        $db = Db::getInstance();
        foreach ($tags as $tag) {
            $tagSql = pSQL($tag);
            $idTag = (int)$db->getValue('SELECT id_tag FROM '._DB_PREFIX_.'tag WHERE name="'.$tagSql.'" AND id_lang='.(int)$idLang);
            if (!$idTag) {
                $db->insert('tag', ['id_lang' => (int)$idLang, 'name' => $tagSql]);
                $idTag = (int)$db->Insert_ID();
            }
            if ($idTag) {
                // éviter doublons produit/tag
                $exists = (int)$db->getValue('SELECT COUNT(*) FROM '._DB_PREFIX_.'product_tag WHERE id_product='.(int)$idProduct.' AND id_tag='.(int)$idTag);
                if (!$exists) {
                    $db->insert('product_tag', ['id_product' => (int)$idProduct, 'id_tag' => (int)$idTag]);
                }
            }
        }
        return true;
    }

    /**
     * Crée un prix spécifique (promo) simple.
     */
    protected function createSpecificPrice(Product $product, string $type, float $value, int $durationDays, int $idShop): bool
    {
        if ($value <= 0) return false;

        $from = date('Y-m-d H:i:s');
        $to = date('Y-m-d H:i:s', strtotime('+'.$durationDays.' days'));

        $reductionType = ($type === 'amount') ? 'amount' : 'percentage';
        $reduction = ($reductionType === 'percentage') ? min($value, 100) / 100.0 : $value;

        $sp = new \SpecificPrice();
        $sp->id_product = (int)$product->id;
        $sp->id_product_attribute = 0;
        $sp->id_cart = 0;
        $sp->id_shop = (int)$idShop;
        $sp->id_shop_group = 0;
        $sp->id_currency = 0;
        $sp->id_country = 0;
        $sp->id_group = 0;
        $sp->id_customer = 0;
        $sp->price = -1; // keep base price
        $sp->from_quantity = 1;
        $sp->reduction = (float)$reduction;
        $sp->reduction_type = $reductionType;
        $sp->from = $from;
        $sp->to = $to;

        return (bool)$sp->add();
    }

    /**
     * Recherche produit par nom approx OU par ID direct.
     * Retourne une petite liste ordonnée (match exact d’abord, puis LIKE).
     */
    public function find_product(array $args): array
    {
        $ctx    = \Context::getContext();
        $idShop = (int)($args['shopId'] ?? ($ctx->shop ? (int)$ctx->shop->id : (int)\Shop::getContextShopID()));
        $idLang = (int)($args['langId'] ?? ($ctx->language ? (int)$ctx->language->id : (int)\Configuration::get('PS_LANG_DEFAULT')));
        $query  = trim((string)($args['query'] ?? ''));
        $limit  = (int)($args['limit'] ?? 5);

        if ($query === '') {
            return ['items' => []];
        }

        // Si l'utilisateur donne un ID numérique, on tente d'abord l'ID.
        if (ctype_digit($query)) {
            $p = new \Product((int)$query, true, $idLang, $idShop);
            if (\Validate::isLoadedObject($p)) {
                return ['items' => [[
                    'id_product' => (int)$p->id,
                    'name'       => (string)$p->name,
                    'active'     => (bool)$p->active,
                ]]];
            }
        }

        $db = \Db::getInstance(_PS_USE_SQL_SLAVE_);
        $q  = pSQL($query);
        // 1) match exact (prioritaire)
        $sqlExact = '
            SELECT p.id_product, pl.name, ps.active
            FROM '._DB_PREFIX_.'product p
            INNER JOIN '._DB_PREFIX_.'product_shop ps ON (ps.id_product=p.id_product AND ps.id_shop='.(int)$idShop.')
            INNER JOIN '._DB_PREFIX_.'product_lang pl ON (pl.id_product=p.id_product AND pl.id_lang='.(int)$idLang.' AND pl.id_shop=ps.id_shop)
            WHERE pl.name = "'.$q.'"
            LIMIT '.(int)$limit;
        $rows = $db->executeS($sqlExact) ?: [];

        // 2) sinon LIKE
        if (count($rows) < $limit) {
            $remain = $limit - count($rows);
            $sqlLike = '
                SELECT p.id_product, pl.name, ps.active
                FROM '._DB_PREFIX_.'product p
                INNER JOIN '._DB_PREFIX_.'product_shop ps ON (ps.id_product=p.id_product AND ps.id_shop='.(int)$idShop.')
                INNER JOIN '._DB_PREFIX_.'product_lang pl ON (pl.id_product=p.id_product AND pl.id_lang='.(int)$idLang.' AND pl.id_shop=ps.id_shop)
                WHERE pl.name LIKE "%'.$q.'%"
                ORDER BY LENGTH(pl.name) ASC
                LIMIT '.(int)$remain;
            $likeRows = $db->executeS($sqlLike) ?: [];
            // éviter doublons si exact déjà trouvé
            $seen = [];
            foreach ($rows as $r) { $seen[(int)$r['id_product']] = true; }
            foreach ($likeRows as $r) {
                if (!isset($seen[(int)$r['id_product']])) $rows[] = $r;
            }
        }

        $items = [];
        foreach ($rows as $r) {
            $items[] = [
                'id_product' => (int)$r['id_product'],
                'name'       => (string)$r['name'],
                'active'     => (bool)$r['active'],
            ];
        }
        return ['items' => $items];
    }

    /**
     * find_product_by_name: retrouve des produits par nom approx ou par ID numérique.
     * Entrée: ['name'=>string, 'langId'?, 'shopId'?, 'limit'?]
     * Sortie: ['items'=>[{'id_product', 'name', 'active'}...]]
     */
    public function find_product_by_name(array $args): array
    {
        $ctx    = \Context::getContext();
        $idShop = (int)($args['shopId'] ?? ($ctx->shop ? (int)$ctx->shop->id : (int)\Shop::getContextShopID()));
        $idLang = (int)($args['langId'] ?? ($ctx->language ? (int)$ctx->language->id : (int)\Configuration::get('PS_LANG_DEFAULT')));
        $name   = trim((string)($args['name'] ?? ''));
        $limit  = (int)($args['limit'] ?? 5);

        if ($name === '') {
            return ['items' => []];
        }

        // Si l'utilisateur a donné un entier → tentative par ID direct (prioritaire)
        if (ctype_digit($name)) {
            $p = new \Product((int)$name, true, $idLang, $idShop);
            if (\Validate::isLoadedObject($p)) {
                return ['items' => [[
                    'id_product' => (int)$p->id,
                    'name'       => (string)$p->name,
                    'active'     => (bool)$p->active,
                ]]];
            }
        }

        $db = \Db::getInstance(_PS_USE_SQL_SLAVE_);
        $q  = pSQL($name);

        // 1) match exact (on met en tête si trouvé)
        $rows = $db->executeS('
            SELECT p.id_product, pl.name, ps.active
            FROM '._DB_PREFIX_.'product p
            INNER JOIN '._DB_PREFIX_.'product_shop ps ON (ps.id_product=p.id_product AND ps.id_shop='.(int)$idShop.')
            INNER JOIN '._DB_PREFIX_.'product_lang pl ON (pl.id_product=p.id_product AND pl.id_lang='.(int)$idLang.' AND pl.id_shop=ps.id_shop)
            WHERE pl.name = "'.$q.'"
            LIMIT '.(int)$limit
        ) ?: [];

        // 2) sinon LIKE (complète jusqu’au limit)
        if (count($rows) < $limit) {
            $remain = $limit - count($rows);
            $like   = $db->executeS('
                SELECT p.id_product, pl.name, ps.active
                FROM '._DB_PREFIX_.'product p
                INNER JOIN '._DB_PREFIX_.'product_shop ps ON (ps.id_product=p.id_product AND ps.id_shop='.(int)$idShop.')
                INNER JOIN '._DB_PREFIX_.'product_lang pl ON (pl.id_product=p.id_product AND pl.id_lang='.(int)$idLang.' AND pl.id_shop=ps.id_shop)
                WHERE pl.name LIKE "%'.$q.'%"
                ORDER BY LENGTH(pl.name) ASC
                LIMIT '.(int)$remain
            ) ?: [];

            // évite les doublons si exact déjà présent
            $seen = [];
            foreach ($rows as $r) { $seen[(int)$r['id_product']] = true; }
            foreach ($like as $r) {
                if (!isset($seen[(int)$r['id_product']])) $rows[] = $r;
            }
        }

        $items = [];
        foreach ($rows as $r) {
            $items[] = [
                'id_product' => (int)$r['id_product'],
                'name'       => (string)$r['name'],
                'active'     => (bool)$r['active'],
            ];
        }
        return ['items' => $items];
    }

    /**
     * Génère une image à partir d’une requête produit (nom ou ID) + choix d’image source.
     * preferredImage: "cover" (défaut) | "first" | "last" | "id:<imageId>"
     */
    public function generate_new_image_by_query(array $args): array
    {
        $ctx    = \Context::getContext();
        $idShop = (int)($args['shopId'] ?? ($ctx->shop ? (int)$ctx->shop->id : (int)\Shop::getContextShopID()));
        $idLang = (int)($args['langId'] ?? ($ctx->language ? (int)$ctx->language->id : (int)\Configuration::get('PS_LANG_DEFAULT')));
        $query  = trim((string)($args['query'] ?? ''));
        $prompt = trim((string)($args['prompt'] ?? ''));
        $pref   = trim((string)($args['preferredImage'] ?? 'cover'));
        $setCov = !empty($args['setAsCover']);

        if ($query === '' || $prompt === '') {
            throw new \RuntimeException('missing_query_or_prompt');
        }

        // 1) Résoudre le produit (ID direct ou par nom)
        $idProduct = 0;
        if (ctype_digit($query)) {
            $idProduct = (int)$query;
        } else {
            $found = $this->find_product(['query' => $query, 'langId' => $idLang, 'shopId' => $idShop, 'limit' => 1]);
            $idProduct = isset($found['items'][0]['id_product']) ? (int)$found['items'][0]['id_product'] : 0;
        }
        if ($idProduct <= 0) {
            throw new \RuntimeException('product_not_found_for_query');
        }

        // 2) Choisir l’image source
        $idImage = $this->pickProductImageId($idProduct, $idLang, $idShop, $pref);
        if ($idImage <= 0) {
            throw new \RuntimeException('no_source_image_found');
        }

        // 3) Re-usage de ton pipeline existant (appel API externe)
        return $this->generateNewImageFromExisting($idProduct, $idImage, $prompt, $setCov);
    }

    /**
     * Sélectionne une image selon la préférence.
     * $preferred peut être "cover", "first", "last" ou "id:<imageId>"
     */
    private function pickProductImageId(int $idProduct, int $idLang, int $idShop, string $preferred): int
    {
        $preferred = strtolower(trim($preferred));
        if (strpos($preferred, 'id:') === 0) {
            $raw = (int)substr($preferred, 3);
            return $raw > 0 ? $raw : 0;
        }

        $rows = \Image::getImages($idLang, $idProduct);
        if (empty($rows)) return 0;

        // Normalise: cover d’abord
        $coverId = 0;
        $firstId = 0;
        $lastId  = 0;
        foreach ($rows as $i => $r) {
            $imgId = (int)$r['id_image'];
            if ($i === 0) $firstId = $imgId;
            $lastId = $imgId;
            if (!empty($r['cover'])) $coverId = $imgId;
        }

        switch ($preferred) {
            case 'first':  return $firstId ?: $coverId ?: $lastId;
            case 'last':   return $lastId  ?: $coverId ?: $firstId;
            case 'cover':
            default:       return $coverId ?: $firstId ?: $lastId;
        }
    }

        /** set_product_active: active/désactive un produit */
    public function set_product_active(array $args): array
    {
        $idProduct = (int)($args['productId'] ?? 0);
        $active    = (bool)($args['active'] ?? false);
        $shopId    = (int)($args['shopId'] ?? (int)Context::getContext()->shop->id);

        $product = new Product($idProduct, false, null, $shopId);
        if (!Validate::isLoadedObject($product)) {
            return ['ok'=>false, 'error'=>'Product not found'];
        }
        $product->active = $active;
        $product->setFieldsToUpdate(['active'=>true]);

        return ['ok'=>(bool)$product->update()];
    }

    /** set_product_price: met à jour le prix HT */
    public function set_product_price(array $args): array
    {
        $idProduct = (int)($args['productId'] ?? 0);
        $price     = (float)($args['price'] ?? -1);
        $shopId    = (int)($args['shopId'] ?? (int)Context::getContext()->shop->id);

        if ($price <= 0) {
            return ['ok'=>false,'error'=>'Invalid price'];
        }
        $product = new Product($idProduct, false, null, $shopId);
        if (!Validate::isLoadedObject($product)) {
            return ['ok'=>false,'error'=>'Product not found'];
        }
        $product->price = $price;
        $product->setFieldsToUpdate(['price'=>true]);

        return ['ok'=>(bool)$product->update()];
    }

    /** set_product_cover_from_first: met la première image en cover */
    public function set_product_cover_from_first(array $args): array
    {
        $idProduct = (int)($args['productId'] ?? 0);
        $shopId    = (int)($args['shopId'] ?? (int)Context::getContext()->shop->id);
        $langId    = (int)Context::getContext()->language->id;

        $product = new Product($idProduct, false, $langId, $shopId);
        if (!Validate::isLoadedObject($product)) {
            return ['ok'=>false,'error'=>'Product not found'];
        }
        $images = $product->getImages($langId);
        if (empty($images)) {
            return ['ok'=>false,'error'=>'No images found'];
        }
        $first = (int)$images[0]['id_image'];
        $ok = $this->setCoverImage($idProduct, $first, $shopId);

        return ['ok'=>$ok,'id_image'=>$first];
    }

    /** set_product_cover: cover spécifique */
    public function set_product_cover(array $args): array
    {
        $idProduct = (int)($args['productId'] ?? 0);
        $idImage   = (int)($args['imageId'] ?? 0);
        $shopId    = (int)($args['shopId'] ?? (int)Context::getContext()->shop->id);

        if ($idProduct<=0 || $idImage<=0) {
            return ['ok'=>false,'error'=>'Missing productId or imageId'];
        }
        $ok = $this->setCoverImage($idProduct, $idImage, $shopId);
        return ['ok'=>$ok,'id_image'=>$idImage];
    }

    /** update_product_name */
    public function update_product_name(array $args): array
    {
        $idProduct = (int)$args['productId'];
        $idLang    = (int)$args['langId'];
        $name     = trim((string)$args['name']);

        $product = new Product($idProduct, false, $idLang);
        if (!Validate::isLoadedObject($product)) {
            return ['ok'=>false,'error'=>'Product not found'];
        }
        $product->name = $name;

        return ['ok'=>(bool)$product->update()];
    }

    /** update_product_description */
    public function update_product_description(array $args): array
    {
        $idProduct = (int)$args['productId'];
        $idLang    = (int)$args['langId'];
        $desc      = trim((string)$args['description']);

        $product = new Product($idProduct, false, $idLang);
        if (!Validate::isLoadedObject($product)) {
            return ['ok'=>false,'error'=>'Product not found'];
        }
        $product->description = $desc;

        return ['ok'=>(bool)$product->update()];
    }

    /** update_product_short_description */
    public function update_product_short_description(array $args): array
    {
        $idProduct = (int)$args['productId'];
        $idLang    = (int)$args['langId'];
        $short     = (string)$args['description_short'];

        $product = new Product($idProduct, false, $idLang);
        if (!Validate::isLoadedObject($product)) {
            return ['ok'=>false,'error'=>'Product not found'];
        }
        $product->description_short = [$idLang=>$short];
        $product->setFieldsToUpdate(['description_short'=>true]);

        return ['ok'=>(bool)$product->update()];
    }

    /** add_product_tags */
    public function add_product_tags(array $args): array
    {
        $idProduct = (int)$args['productId'];
        $idLang    = (int)$args['langId'];
        $tags      = $args['tags'] ?? [];

        $ok = $this->addTags($idProduct, $idLang, $tags);
        return ['ok'=>$ok];
    }

    /** create_specific_price (promo simple) */
    public function create_specific_price(array $args): array
    {
        $idProduct = (int)$args['productId'];
        $type      = (string)$args['type'];
        $value     = (float)$args['value'];
        $days      = (int)($args['durationDays'] ?? 14);
        $shopId    = (int)($args['shopId'] ?? (int)Context::getContext()->shop->id);

        $product = new Product($idProduct, false, null, $shopId);
        if (!Validate::isLoadedObject($product)) {
            return ['ok'=>false,'error'=>'Product not found'];
        }
        $ok = $this->createSpecificPrice($product,$type,$value,$days,$shopId);
        return ['ok'=>$ok];
    }

    /** ensure_variant_images (placeholder) */
    public function ensure_variant_images(array $args): array
    {
        return ['ok'=>true,'note'=>'Variant images check TODO'];
    }

    /** review_price_positioning (placeholder) */
    public function review_price_positioning(array $args): array
    {
        return ['ok'=>true,'note'=>'Price positioning audit TODO'];
    }

    /** request_new_images (placeholder) */
    public function request_new_images(array $args): array
    {
        $min = (int)($args['minImages'] ?? 3);
        return ['ok'=>true,'note'=>'Request '.$min.' new images TODO'];
    }

    /** resolve_backoffice_location (placeholder simple) */
    public function resolve_backoffice_location(array $args): array
    {
        $intent = (string)$args['intent'];
        // Ex: mapping basique
        $map = [
            'factures' => ['url'=>'index.php?controller=AdminInvoices','selector'=>'#form-invoices'],
            'produits' => ['url'=>'index.php?controller=AdminProducts','selector'=>'#product_catalog_list'],
        ];
        foreach ($map as $k=>$v) {
            if (stripos($intent,$k)!==false) {
                return ['relativeUrl'=>$v['url'],'selector'=>$v['selector']];
            }
        }
        return ['relativeUrl'=>'index.php?controller=AdminDashboard','selector'=>null];
    }

    /** open_backoffice_location (passe la requête au front via JS/iframe) */
    public function open_backoffice_location(array $args): array
    {
        return [
            'ok'=>true,
            'relativeUrl'=>(string)$args['relativeUrl'],
            'selector'=>$args['selector'] ?? null,
            'fragment'=>$args['fragment'] ?? null
        ];
    }

    /**
     * Retour principal pour le LLM : toutes les informations d'un produit.
     *
     * @param int $productId
     * @param int|null $langId
     * @param int|null $shopId
     * @param array $opts  // ['sales_from'=>'YYYY-MM-DD','sales_to'=>'YYYY-MM-DD']
     * @return array
     * @throws \RuntimeException
     */
    public function getProductInfo(int $productId, ?int $langId = null, ?int $shopId = null, array $opts = []): array
    {
        $ctx = Context::getContext();

        $shopId = $shopId ?? ($ctx->shop ? (int)$ctx->shop->id : (int)\Shop::getContextShopID());
        $langId = $langId ?? ($ctx->language ? (int)$ctx->language->id : (int)Configuration::get('PS_LANG_DEFAULT'));

        $product = new Product($productId, true, $langId, $shopId);
        if (!Validate::isLoadedObject($product)) {
            throw new \RuntimeException('Product not found: ' . $productId);
        }

        // basic fields
        $name = is_array($product->name) ? ($product->name[$langId] ?? reset($product->name)) : (string)$product->name;
        $linkRewrite = $this->getProductLinkRewrite($product, $langId);

        // images
        $images = $this->listImagesForProduct($productId, $langId, $shopId);

        // stock
        $stockQty = $this->getStockQty($productId, $shopId);

        // categories (list & default)
        $categories = $this->getProductCategories($productId, $langId, $shopId);
        $defaultCategoryId = (int)$product->id_category_default;
        $defaultCategory = $categories['by_id'][$defaultCategoryId] ?? null;

        // tags
        $tags = $this->getProductTags($productId);

        // attributes / combinations
        $attributes = $this->getProductAttributesSummary($productId, $langId, $shopId);

        // sales stats (optional period)
        $salesFrom = $opts['sales_from'] ?? date('Y-m-01');
        $salesTo   = $opts['sales_to']   ?? date('Y-m-d');
        $salesStats = $this->getProductSalesStats($productId, $shopId, $salesFrom, $salesTo);

        // category median/avg price reference
        $medianPriceByCategory = $this->getCategoryAveragePrices(array_keys($categories['by_id']), $shopId, $langId);

        // extra metadata
        $metaTitle = $this->getLangField($product, 'meta_title', $langId);
        $metaDescription = $this->getLangField($product, 'meta_description', $langId);
        $descriptionShort = $this->getLangField($product, 'description_short', $langId);
        $description = $this->getLangField($product, 'description', $langId);

        // build output
        $out = [
            'id_product' => (int)$productId,
            'name' => (string)$name,
            'link_rewrite' => (string)$linkRewrite,
            'reference' => (string)$product->reference,
            'ean13' => (string)$product->ean13,
            'price' => (float)$product->price, // HT
            'active' => (bool)$product->active,
            'visibility' => (string)($product->visibility ?? ''),
            'condition' => (string)($product->condition ?? ''),
            'weight' => (float)($product->weight ?? 0),
            'date_add' => (string)($product->date_add ?? ''),
            'date_upd' => (string)($product->date_upd ?? ''),

            // SEO & descriptions
            'meta_title' => (string)$metaTitle,
            'meta_description' => (string)$metaDescription,
            'description_short' => (string)$descriptionShort,
            'description' => (string)$description,

            // images
            'images' => $images['items'],
            'image_count' => (int)$images['count'],
            'has_cover' => (bool)$images['has_cover'],
            'cover_image_url' => $images['cover_url'] ?? null,

            // stock & variants
            'stock_qty' => (int)$stockQty,
            'has_combinations' => !empty($attributes['combinations']),
            'combinations' => $attributes['combinations'],
            'attributes_summary' => $attributes['summary'],

            // categories/tags
            'categories' => array_values($categories['list']),
            'categories_by_id' => $categories['by_id'],
            'default_category' => $defaultCategory,
            'tags' => $tags,

            // sales
            'sales' => $salesStats,
            'category_price_ref' => $medianPriceByCategory,

            // simple SEO audit
            'seo_audit' => $this->computeSeoAudit([
                'meta_title' => $metaTitle,
                'meta_description' => $metaDescription,
                'description_short' => $descriptionShort,
                'description' => $description,
                'image_count' => $images['count'],
                'has_cover' => $images['has_cover'],
            ]),
        ];

        return $out;
    }

    /**
     * Liste les images avec URLs prêtes à l'emploi (large_default).
     *
     * @return array ['items'=>[...], 'count'=>int, 'has_cover'=>bool, 'cover_url'=>string|null]
     */
    protected function listImagesForProduct(int $productId, int $langId, int $shopId): array
    {
        $rows = Image::getImages($langId, $productId) ?: [];
        $items = [];
        $coverUrl = null;
        $hasCover = false;

        $product = new Product($productId, true, $langId, $shopId);
        $linkRewrite = $this->getProductLinkRewrite($product, $langId);

        foreach ($rows as $r) {
            $idImage = (int)$r['id_image'];
            $isCover = !empty($r['cover']);
            $url = Context::getContext()->link->getImageLink($linkRewrite, $idImage, 'large_default');
            $items[] = [
                'id_image' => $idImage,
                'position' => isset($r['position']) ? (int)$r['position'] : 0,
                'cover' => (bool)$isCover,
                'url' => $url,
            ];
            if ($isCover) {
                $hasCover = true;
                $coverUrl = $url;
            }
        }

        return ['items' => $items, 'count' => count($items), 'has_cover' => $hasCover, 'cover_url' => $coverUrl];
    }

    protected function getStockQty(int $productId, int $shopId): int
    {
        // Utilise l’API PrestaShop quand elle est disponible
        if (method_exists('\\StockAvailable', 'getQuantityAvailableByProduct')) {
            // id_product_attribute = 0 (produit “simple” au global), $shopId au besoin
            $q = \StockAvailable::getQuantityAvailableByProduct($productId, 0, $shopId);
            return (int) $q;
        }

        // Fallback SQL — sans LIMIT, via agrégat (évite les soucis MariaDB)
        $db = \Db::getInstance();

        // 1) quantité pour le shop précis OU ligne globale (id_shop=0)
        $qty = $db->getValue('
            SELECT MAX(quantity) 
            FROM '._DB_PREFIX_.'stock_available
            WHERE id_product='.(int)$productId.'
            AND id_product_attribute=0
            AND (id_shop='.(int)$shopId.' OR id_shop=0)
        ');
        if ($qty !== false && $qty !== null) {
            return (int) $qty;
        }

        // 2) dernier recours
        $qty = $db->getValue('
            SELECT MAX(quantity)
            FROM '._DB_PREFIX_.'stock_available
            WHERE id_product='.(int)$productId.' AND id_product_attribute=0
        ');
        return ($qty === false || $qty === null) ? 0 : (int) $qty;
    }

    protected function getProductCategories(int $productId, int $langId, int $shopId): array
    {
        $db = Db::getInstance(_PS_USE_SQL_SLAVE_);
        $sql = 'SELECT cp.id_category, cl.name, cp.position
                FROM '._DB_PREFIX_.'category_product cp
                INNER JOIN '._DB_PREFIX_.'category_lang cl ON cl.id_category = cp.id_category AND cl.id_lang = '.(int)$langId.'
                WHERE cp.id_product = '.(int)$productId.'
                ORDER BY cp.position ASC';
        $rows = $db->executeS($sql) ?: [];

        $list = []; $byId = [];
        foreach ($rows as $r) {
            $id = (int)$r['id_category'];
            $entry = ['id_category'=>$id, 'name'=>$r['name'] ?? '', 'position'=>(int)$r['position']];
            $list[] = $entry;
            $byId[$id] = $entry;
        }
        return ['list'=>$list, 'by_id'=>$byId];
    }

    protected function getProductTags(int $productId): array
    {
        $db = Db::getInstance(_PS_USE_SQL_SLAVE_);
        $rows = $db->executeS('
            SELECT t.id_tag, t.name, t.id_lang
            FROM '._DB_PREFIX_.'tag t
            INNER JOIN '._DB_PREFIX_.'product_tag pt ON pt.id_tag = t.id_tag
            WHERE pt.id_product = '.(int)$productId
        ) ?: [];
        $out = [];
        foreach ($rows as $r) {
            $out[] = ['id_tag' => (int)$r['id_tag'], 'name' => (string)$r['name'], 'lang' => (int)$r['id_lang']];
        }
        return $out;
    }

    /**
     * Récupère un résumé des combinaisons (attributs) d’un produit.
     */
    protected function getProductAttributesSummary(int $productId, int $langId, int $shopId): array
    {
        $combs = [];
        $summary = [];

        // Instancie le produit dans le bon contexte
        $product = new \Product((int) $productId, false, (int) $langId, (int) $shopId);

        if (method_exists($product, 'getAttributeCombinations')) {
            // getAttributeCombinations accepte généralement ($idLang, $groupByIdAttribute = false)
            $raw = (array) $product->getAttributeCombinations((int) $langId, true);

            foreach ($raw as $r) {
                $combId = (int)($r['id_product_attribute'] ?? 0);
                $combs[$combId][] = [
                    'group'       => $r['group_name']     ?? '',
                    'attribute'   => $r['attribute_name'] ?? '',
                    'id_attribute'=> (int)($r['id_attribute'] ?? 0),
                ];
            }

            // résumé
            $summary = [
                'combinations_count' => count($combs),
                'attributes_count'   => count($raw),
            ];
        }

        return [
            'combinations' => $combs,
            'summary'      => $summary,
        ];
    }

    /**
     * Ventes pour UN produit sur une période (date_from inclusive, date_to exclusive)
     * Retour: ['qty' => int, 'revenue' => float]
     */
    public function getProductSalesStats(int $productId, int $shopId, string $dateFrom, string $dateTo): array
    {
        $db = Db::getInstance(_PS_USE_SQL_SLAVE_);
        $sql = '
            SELECT SUM(od.product_quantity) AS qty, SUM(od.total_price_tax_incl) AS revenue
            FROM '._DB_PREFIX_.'order_detail od
            INNER JOIN '._DB_PREFIX_.'orders o ON o.id_order = od.id_order
            WHERE od.product_id = '.(int)$productId.'
              AND o.date_add >= "'.pSQL($dateFrom).'"
              AND o.date_add < DATE_ADD("'.pSQL($dateTo).'", INTERVAL 1 DAY)
              AND o.valid = 1
              AND o.id_shop = '.(int)$shopId;
        $row = $db->getRow($sql);
        return [
            'qty' => (int)($row['qty'] ?? 0),
            'revenue' => round((float)($row['revenue'] ?? 0.0), 2),
            'period' => ['from'=>$dateFrom,'to'=>$dateTo],
        ];
    }

    /**
     * Calcule une moyenne des prix pour une liste de catégories (référence)
     * Retour: [id_category => avg_price, ...]
     */
    protected function getCategoryAveragePrices(array $catIds, int $shopId, int $langId): array
    {
        $out = [];
        if (empty($catIds)) return $out;
        $db = Db::getInstance(_PS_USE_SQL_SLAVE_);
        foreach ($catIds as $idCat) {
            $sql = '
                SELECT AVG(ps.price) AS avg_price
                FROM '._DB_PREFIX_.'product_shop ps
                INNER JOIN '._DB_PREFIX_.'category_product cp ON cp.id_product = ps.id_product
                WHERE ps.id_shop = '.(int)$shopId.' AND cp.id_category = '.(int)$idCat;
            $avg = $db->getValue($sql);
            $out[(int)$idCat] = $avg === false ? null : (float)$avg;
        }
        return $out;
    }

    /**
     * Heuristiques SEO très légères (pour donner un diagnostic au LLM)
     *
     * - title length, meta length, short desc length, images count, cover presence
     * - renvoie score 0..100 et liste d'issues
     */
    public function computeSeoAudit(array $data): array
    {
        $title = trim((string)($data['meta_title'] ?? ''));
        $meta = trim((string)($data['meta_description'] ?? ''));
        $short = trim((string)($data['description_short'] ?? ''));
        $desc = trim((string)($data['description'] ?? ''));
        $imgCount = (int)($data['image_count'] ?? 0);
        $hasCover = !empty($data['has_cover']);

        $score = 100;
        $issues = [];

        // Title heuristics
        $tlen = mb_strlen($title);
        if ($tlen === 0) { $score -= 30; $issues[] = 'meta_title missing'; }
        elseif ($tlen < 30) { $score -= 10; $issues[] = 'meta_title too short'; }
        elseif ($tlen > 70) { $score -= 5; $issues[] = 'meta_title maybe too long'; }

        // Meta description heuristics
        $mlen = mb_strlen($meta);
        if ($mlen === 0) { $score -= 20; $issues[] = 'meta_description missing'; }
        elseif ($mlen < 50) { $score -= 10; $issues[] = 'meta_description too short'; }
        elseif ($mlen > 320) { $score -= 5; $issues[] = 'meta_description maybe too long'; }

        // Short desc
        $slen = mb_strlen(strip_tags($short));
        if ($slen < 60) { $score -= 10; $issues[] = 'short_description too short'; }

        // Images
        if (!$hasCover) { $score -= 10; $issues[] = 'no cover image'; }
        if ($imgCount < 3) { $score -= 5; $issues[] = 'few images (<3)'; }

        // Description
        $dlen = mb_strlen(strip_tags($desc));
        if ($dlen < 200) { $score -= 5; $issues[] = 'description short (suggest longer product story)'; }

        if ($score < 0) $score = 0;
        if ($score > 100) $score = 100;

        return ['score' => (int)$score, 'issues' => $issues];
    }

    // ---------- helpers ----------
    private function getLangField(Product $product, string $field, int $idLang): string
    {
        $val = $product->{$field} ?? '';
        if (is_array($val)) {
            return (string)($val[$idLang] ?? reset($val));
        }
        return (string)$val;
    }

    public function update_product_field(array $args): array
    {
        $productId = (int) ($args['productId'] ?? 0);
        $langId    = (int) ($args['langId']    ?? (int) \Context::getContext()->language->id);
        $shopId    = (int) ($args['shopId']    ?? (int) \Context::getContext()->shop->id);
        $key       = (string) ($args['key']    ?? '');
        $value     = $args['value'] ?? null;

        if ($productId <= 0 || $key === '') {
            return ['ok' => false, 'error' => 'Missing productId or key'];
        }

        try {
            if (\Shop::isFeatureActive()) {
                \Shop::setContext(\Shop::CONTEXT_SHOP, $shopId);
            }

            // Charger toutes les langues pour avoir des tableaux lang
            $product = new \Product($productId, true, null, $shopId);
            if (!\Validate::isLoadedObject($product)) {
                return ['ok' => false, 'error' => 'Product not found'];
            }

            $def = \Product::$definition['fields'] ?? [];
            if (!isset($def[$key])) {
                return ['ok' => false, 'error' => 'Unsupported or unknown field: '.$key];
            }

            // Typage léger
            $typed = $value;
            switch ($def[$key]['type'] ?? null) {
                case \ObjectModel::TYPE_INT:   $typed = (int) $value;  break;
                case \ObjectModel::TYPE_BOOL:  $typed = (bool)$value;  break;
                case \ObjectModel::TYPE_FLOAT: $typed = (float)$value; break;
                default: /* string/html/sql */                         break;
            }

            $isLang = !empty($def[$key]['lang']);

            // Snapshot
            $previous = $isLang
                ? (is_array($product->{$key}) ? ($product->{$key}[$langId] ?? null) : $product->{$key})
                : ($product->{$key} ?? null);

            if ($isLang) {
                // Mettre à jour uniquement la langue ciblée sans casser les autres
                $vals = is_array($product->{$key}) ? $product->{$key} : [];
                $vals[$langId] = (string)$typed;
                $product->{$key} = $vals;

                // Si on modifie le name → régénérer le link_rewrite de la même langue
                if ($key === 'name') {
                    $title = (string)$vals[$langId];
                    $slug = method_exists(\Tools::class, 'link_rewrite')
                        ? (string)\Tools::link_rewrite($title)
                        : (method_exists(\Tools::class, 'str2url') ? (string)\Tools::str2url($title) : '');
                    if ($slug === '') { $slug = 'product'; }

                    $lr = is_array($product->link_rewrite) ? $product->link_rewrite : [];
                    $lr[$langId] = $slug;
                    $product->link_rewrite = $lr;
                }
            } else {
                // Champ non multilingue
                $product->{$key} = $typed;
            }

            // Cibler la boutique (multishop)
            $product->id_shop_list = [$shopId];

            // Validations (facultatif mais utile)
            $e1 = $product->validateFields(false, true);
            $e2 = $product->validateFieldsLang(false, true);
            if ($e1 !== true || $e2 !== true) {
                $errs = [];
                if ($e1 !== true) $errs[] = $e1;
                if ($e2 !== true) $errs[] = $e2;
                return ['ok' => false, 'error' => 'Validation failed', 'details' => $errs];
            }

            // ✅ Pas de setFieldsToUpdate() → on laisse l’ORM écrire correctement product & product_lang
            if (!$product->update()) {
                return ['ok' => false, 'error' => 'update_failed'];
            }

            // Relecture propre (langue demandée)
            $refreshed = new \Product($productId, false, $langId, $shopId);
            $current = $refreshed->{$key} ?? null;

            return [
                'ok'   => true,
                'data' => [
                    'productId' => $productId,
                    'updated'   => [$key],
                    'previous'  => $previous,
                    'current'   => $current,
                ],
            ];
        } catch (\Exception $e) {
            return ['ok' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Petit helper: détecte si un tableau est associatif.
     */
    private function isAssoc(array $arr): bool
    {
        if ($arr === []) return false;
        return array_keys($arr) !== range(0, count($arr) - 1);
    }

}
