<?php
namespace Iassistant\Service;

use Context;
use Iassistant\Mcp\McpTools;

class McpExecutor
{
    /** @var McpTools */
    private $tools;

    public function __construct(Context $context)
    {
        $this->tools = new McpTools();
    }

    /**
     * Exécute un tool MCP exposé par Iassistant\Mcp\McpTools
     *
     * @param string $tool  Nom du tool (traits d’union acceptés, insensible à la casse)
     * @param array  $args  Arguments du tool
     * @return array
     */
    public function execute(string $tool, array $args)
    {
        // Normalisation du nom (ex: "set-product-cover-image" -> "set_product_cover_image")
        $tool = str_replace('-', '_', strtolower($tool));

        // petite closure utilitaire pour choisir la meilleure valeur (priorité bool > number > object > array > string)
        $pickValue = function(array $a) {
            if (array_key_exists('value_boolean', $a)) return (bool)$a['value_boolean'];
            if (array_key_exists('value_number',  $a)) return $a['value_number'];
            if (array_key_exists('value_object',  $a)) return $a['value_object'];
            if (array_key_exists('value_array',   $a)) return $a['value_array'];
            return $a['value'] ?? null;
        };

        switch ($tool) {
            // =======================
            // IMAGES
            // =======================
            case 'list_product_images':
                return $this->tools->listProductImages(
                    (int)$this->req($args, 'productId'),
                    (int)$this->req($args, 'langId'),
                    (string)($args['imageType'] ?? 'large_default')
                );

            case 'get_product_image_base64':
                return $this->tools->getProductImageBase64(
                    (int)$this->req($args, 'imageId')
                );

            case 'save_generated_image':
                return $this->tools->saveGeneratedImage(
                    (int)$this->req($args, 'productId'),
                    (string)$this->req($args, 'imageB64'),
                    !empty($args['setAsCover'])
                );

            case 'regenerate_product_image_thumbnails':
                return $this->tools->regenerateProductImageThumbnails(
                    (int)$this->req($args, 'imageId'),
                    (bool)($args['clean'] ?? false)
                );

            case 'set_product_cover_image':
                return $this->tools->setProductCoverImage(
                    (int)$this->req($args, 'productId'),
                    (int)$this->req($args, 'imageId')
                );

            case 'generate_new_image_from_existing':
            case 'generate_product_image_from_existing':
            case 'generate_product_image':
                return $this->tools->generateNewImageFromExisting(
                    (int)$this->req($args, 'productId'),
                    (int)$this->req($args, 'imageId'),
                    (string)$this->req($args, 'prompt'),
                    !empty($args['setAsCover'])
                );

            case 'generate_new_image_by_query':
                return $this->tools->generate_new_image_by_query((array)$args);

            // =======================
            // LANGUES
            // =======================
            case 'get_lang_id':
                return $this->tools->getLangId(
                    isset($args['isoCode']) ? (string)$args['isoCode'] : null,
                    isset($args['locale'])  ? (string)$args['locale']  : null
                );

            case 'list_languages':
                return $this->tools->listLanguages(
                    (bool)($args['activeOnly'] ?? true)
                );

            // =======================
            // PRODUITS
            // =======================
            case 'create_product':
                return $this->tools->createProduct((array)$args);

            case 'add_product_images_base64':
                return $this->tools->addProductImagesBase64(
                    (int)$this->req($args, 'productId'),
                    (array)$this->req($args, 'imagesB64'),
                    !empty($args['setFirstAsCover'])
                );

            // ----- Mappés vers update_product_field -----
            case 'set_product_active':
                return $this->tools->update_product_field([
                    'productId' => (int)$this->req($args, 'productId'),
                    'key'       => 'active',
                    'value'     => isset($args['active']) ? (bool)$args['active'] : true,
                    'langId'    => isset($args['langId']) ? (int)$args['langId'] : (int)\Context::getContext()->language->id,
                    'shopId'    => isset($args['shopId']) ? (int)$args['shopId'] : (int)\Context::getContext()->shop->id,
                ]);

            case 'set_product_price':
                return $this->tools->update_product_field([
                    'productId' => (int)$this->req($args, 'productId'),
                    'key'       => 'price',
                    'value'     => (float)($args['price'] ?? -1),
                    'langId'    => isset($args['langId']) ? (int)$args['langId'] : (int)\Context::getContext()->language->id,
                    'shopId'    => isset($args['shopId']) ? (int)$args['shopId'] : (int)\Context::getContext()->shop->id,
                ]);

            case 'update_product_name':
                return $this->tools->update_product_field([
                    'productId' => (int)$this->req($args, 'productId'),
                    'key'       => 'name',
                    'value'     => (string)$this->req($args, 'name'),
                    'langId'    => isset($args['langId']) ? (int)$args['langId'] : (int)\Context::getContext()->language->id,
                    'shopId'    => isset($args['shopId']) ? (int)$args['shopId'] : (int)\Context::getContext()->shop->id,
                ]);

            case 'update_product_description':
                return $this->tools->update_product_field([
                    'productId' => (int)$this->req($args, 'productId'),
                    'key'       => 'description',
                    'value'     => (string)$this->req($args, 'description'),
                    'langId'    => isset($args['langId']) ? (int)$args['langId'] : (int)\Context::getContext()->language->id,
                    'shopId'    => isset($args['shopId']) ? (int)$args['shopId'] : (int)\Context::getContext()->shop->id,
                ]);

            case 'update_product_short_description':
                return $this->tools->update_product_field([
                    'productId' => (int)$this->req($args, 'productId'),
                    'key'       => 'description_short',
                    'value'     => (string)$this->req($args, 'description_short'),
                    'langId'    => isset($args['langId']) ? (int)$args['langId'] : (int)\Context::getContext()->language->id,
                    'shopId'    => isset($args['shopId']) ? (int)$args['shopId'] : (int)\Context::getContext()->shop->id,
                ]);
            // --------------------------------------------

            case 'set_product_cover_from_first':
                return $this->tools->set_product_cover_from_first((array)$args);

            case 'set_product_cover':
                return $this->tools->set_product_cover((array)$args);

            case 'add_product_tags':
                return $this->tools->add_product_tags((array)$args);

            case 'create_specific_price':
                return $this->tools->create_specific_price((array)$args);

            case 'ensure_variant_images':
                return $this->tools->ensure_variant_images((array)$args);

            case 'review_price_positioning':
                return $this->tools->review_price_positioning((array)$args);

            case 'request_new_images':
                return $this->tools->request_new_images((array)$args);

            // =======================
            // RECHERCHE PRODUIT
            // =======================
            case 'find_product':
                return $this->tools->find_product((array)$args);

            case 'find_product_by_name':
                if (isset($args['name']) && !isset($args['query'])) {
                    $args['query'] = $args['name'];
                }
                return $this->tools->find_product((array)$args);

            // =======================
            // ANALYSE & RECO
            // =======================
            case 'find_low_selling_products':
                return $this->tools->find_low_selling_products((array)$args);

            case 'apply_product_recommendations':
                return $this->tools->apply_product_recommendations((array)$args);

            // =======================
            // BACK-OFFICE
            // =======================
            case 'resolve_backoffice_location':
                return $this->tools->resolve_backoffice_location((array)$args);

            case 'open_backoffice_location':
                return $this->tools->open_backoffice_location((array)$args);

            case 'get_product_info':
                return $this->tools->getProductInfo(
                    (int)$this->req($args,'productId'),
                    isset($args['langId']) ? (int)$args['langId'] : null,
                    isset($args['shopId']) ? (int)$args['shopId'] : null,
                    $args['opts'] ?? []
                );

            // ----- générique : deux alias acceptés -----
            case 'update_field':
            case 'update_product_field':
                return $this->tools->update_product_field([
                    'productId'     => (int) $this->req($args, 'productId'),
                    'key'           => (string) $this->req($args, 'key'),
                    // passe aussi les value_* pour que McpTools gère le merge/multilang
                    'value'         => $args['value']         ?? null,
                    'value_boolean' => $args['value_boolean'] ?? null,
                    'value_number'  => $args['value_number']  ?? null,
                    'value_object'  => $args['value_object']  ?? null,
                    'value_array'   => $args['value_array']   ?? null,
                    'langId'        => isset($args['langId']) ? (int)$args['langId'] : (int)\Context::getContext()->language->id,
                    'shopId'        => isset($args['shopId']) ? (int)$args['shopId'] : (int)\Context::getContext()->shop->id,
                ]);

            default:
                throw new \RuntimeException('Unknown tool: ' . $tool);
        }
    }

    /**
     * Récupère un argument requis ou lève une exception claire.
     */
    private function req(array $args, string $key)
    {
        if (!array_key_exists($key, $args)) {
            throw new \InvalidArgumentException('Missing parameter: ' . $key);
        }
        return $args[$key];
    }
}
